<?php
require_once __DIR__ . '/../config/Database.php';

class Wakur {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    /**
     * Mengambil daftar mata pelajaran yang memiliki soal menunggu validasi.
     * @return array Daftar mapel beserta jumlah soal pending.
     */
     public function getLaporanUjian() {
        $stmt = $this->db->query(
            "SELECT
                u.judul_ujian,
                mp.nama_pelajaran,
                u.created_at,
                (SELECT COUNT(hasil_id) FROM hasil_ujian WHERE ujian_id = u.ujian_id) AS jumlah_peserta,
                (SELECT AVG(skor_akhir) FROM hasil_ujian WHERE ujian_id = u.ujian_id) AS rata_rata_skor
            FROM
                ujian u
            JOIN
                mata_pelajaran mp ON u.mapel_id = mp.mapel_id
            ORDER BY
                u.created_at DESC"
        );
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    public function getMapelWithPendingSoal() {
        $stmt = $this->db->query(
            "SELECT mp.mapel_id, mp.nama_pelajaran, COUNT(s.soal_id) as jumlah_pending
             FROM soal s
             JOIN mata_pelajaran mp ON s.mapel_id = mp.mapel_id
             WHERE s.status_validasi = 'Pending'
             GROUP BY mp.mapel_id, mp.nama_pelajaran
             ORDER BY mp.nama_pelajaran ASC"
        );
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Mengambil semua soal yang statusnya 'Pending' untuk mata pelajaran tertentu.
     * @param int $mapelId ID mata pelajaran.
     * @return array Daftar soal.
     */
    public function getPendingSoalByMapel($mapelId) {
        $stmt = $this->db->prepare(
            "SELECT soal_id, pertanyaan, jenis_soal 
             FROM soal 
             WHERE mapel_id = :mapel_id AND status_validasi = 'Pending'"
        );
        $stmt->execute(['mapel_id' => $mapelId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Memproses validasi untuk semua soal pending dalam satu mata pelajaran.
     * @param int $mapelId ID mata pelajaran.
     * @param string $status Status baru ('Disetujui' atau 'Ditolak').
     * @param string $catatan Catatan dari validator.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function prosesValidasiMapel($mapelId, $status, $catatan) {
        try {
            $stmt = $this->db->prepare(
                "UPDATE soal 
                 SET status_validasi = :status, catatan_validasi = :catatan
                 WHERE mapel_id = :mapel_id AND status_validasi = 'Pending'"
            );
            return $stmt->execute([
                'status' => $status,
                'catatan' => $catatan,
                'mapel_id' => $mapelId
            ]);
        } catch (PDOException $e) {
            error_log("Proses Validasi Mapel Error: " . $e->getMessage());
            return false;
        }
    }
}
?>