<?php
require_once __DIR__ . '/../config/Database.php';

class UjianSiswa {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function getExamHistory($siswaId) {
        try {
            $stmt = $this->db->prepare(
                "SELECT hu.hasil_id, u.judul_ujian, mp.nama_pelajaran, hu.skor_akhir, hu.created_at
                 FROM hasil_ujian hu
                 JOIN ujian u ON hu.ujian_id = u.ujian_id
                 JOIN mata_pelajaran mp ON u.mapel_id = mp.mapel_id
                 WHERE hu.siswa_id = :siswa_id
                 ORDER BY hu.created_at DESC"
            );
            $stmt->execute(['siswa_id' => $siswaId]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }

    public function getAvailableExamsForSiswa($siswaId) {
        try {
            $stmt = $this->db->prepare(
                "SELECT u.ujian_id, u.judul_ujian, u.kode_ujian, mp.nama_pelajaran 
                 FROM ujian u
                 JOIN mata_pelajaran mp ON u.mapel_id = mp.mapel_id
                 WHERE u.ujian_id NOT IN (SELECT ujian_id FROM hasil_ujian WHERE siswa_id = :siswa_id)
                 ORDER BY u.created_at DESC"
            );
            $stmt->execute(['siswa_id' => $siswaId]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }
    
    public function submitAndCalculateScore($ujianId, $siswaId, $answers) {
        try {
            $soalStmt = $this->db->prepare(
                "SELECT s.soal_id, s.jenis_soal, s.jawaban_benar 
                 FROM soal s JOIN ujian_detail ud ON s.soal_id = ud.soal_id
                 WHERE ud.ujian_id = :ujian_id"
            );
            $soalStmt->execute(['ujian_id' => $ujianId]);
            $soalDataFromDB = $soalStmt->fetchAll(PDO::FETCH_ASSOC);

            $kunciJawabanMap = [];
            foreach($soalDataFromDB as $soal) {
                $kunciJawabanMap[$soal['soal_id']] = $soal;
            }

            $skorBenar = 0;
            $jumlahSoalPg = 0;
            $feedbackAI = "Analisis Jawaban Esai:\n";
            $foundEssay = false;

            foreach ($answers as $soalId => $jawabanSiswa) {
                if (isset($kunciJawabanMap[$soalId])) {
                    $soalInfo = $kunciJawabanMap[$soalId];
                    
                    if ($soalInfo['jenis_soal'] == 'Pilihan Ganda') {
                        $jumlahSoalPg++;
                        if ($jawabanSiswa !== null && $soalInfo['jawaban_benar'] === $jawabanSiswa) {
                            $skorBenar++;
                        }
                    } elseif ($soalInfo['jenis_soal'] == 'Esai') {
                        $foundEssay = true;
                        if (empty(trim($jawabanSiswa))) {
                            $feedbackAI .= "- Soal Esai ID #$soalId: Jawaban kosong.\n";
                        } elseif (strlen($jawabanSiswa) < 50) {
                            $feedbackAI .= "- Soal Esai ID #$soalId: Jawaban Anda terlalu singkat. Coba berikan penjelasan yang lebih mendalam dan detail.\n";
                        } elseif (stripos($jawabanSiswa, $soalInfo['jawaban_benar']) !== false) {
                            $feedbackAI .= "- Soal Esai ID #$soalId: Jawaban Anda sudah baik dan mengandung kata kunci yang benar. Pertahankan!\n";
                        } else {
                            $feedbackAI .= "- Soal Esai ID #$soalId: Jawaban sudah cukup deskriptif, namun perlu lebih fokus pada inti pertanyaan sesuai kunci jawaban.\n";
                        }
                    }
                }
            }
            if (!$foundEssay) {
                $feedbackAI = "Tidak ada soal esai pada ujian ini.";
            }

            $skorAkhir = ($jumlahSoalPg > 0) ? ($skorBenar / $jumlahSoalPg) * 100 : 0;
            $skorAkhir = round($skorAkhir, 2);

            $insertStmt = $this->db->prepare(
                "INSERT INTO hasil_ujian (ujian_id, siswa_id, skor_akhir, jawaban_siswa, feedback_ai) 
                 VALUES (:ujian_id, :siswa_id, :skor_akhir, :jawaban_siswa, :feedback)"
            );
            $insertStmt->execute([
                'ujian_id' => $ujianId,
                'siswa_id' => $siswaId,
                'skor_akhir' => $skorAkhir,
                'jawaban_siswa' => json_encode($answers),
                'feedback' => $feedbackAI
            ]);
            
            return $this->db->lastInsertId();

        } catch (PDOException $e) {
            error_log("Submit Exam Error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getExamResult($hasilId) {
        try {
            $stmt = $this->db->prepare(
                "SELECT hu.ujian_id, u.judul_ujian, us.name AS nama_siswa, hu.skor_akhir, hu.jawaban_siswa, hu.feedback_ai, hu.created_at
                 FROM hasil_ujian hu
                 JOIN ujian u ON hu.ujian_id = u.ujian_id
                 JOIN siswa s ON hu.siswa_id = s.siswa_id
                 JOIN users us ON s.user_id = us.user_id
                 WHERE hu.hasil_id = :hasil_id"
            );
            $stmt->execute(['hasil_id' => $hasilId]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return null;
        }
    }

    public function getExamDetails($ujianId) {
        try {
            $sql = "SELECT 
                        u.ujian_id, u.judul_ujian, 
                        s.soal_id, s.pertanyaan, s.jenis_soal, s.jawaban_benar
                    FROM ujian u
                    JOIN ujian_detail ud ON u.ujian_id = ud.ujian_id
                    JOIN soal s ON ud.soal_id = s.soal_id
                    WHERE u.ujian_id = :ujian_id";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute(['ujian_id' => $ujianId]);
            $soalList = $stmt->fetchAll(PDO::FETCH_ASSOC);

            foreach ($soalList as $key => $soal) {
                if ($soal['jenis_soal'] == 'Pilihan Ganda') {
                    $opsiStmt = $this->db->prepare("SELECT opsi_text FROM opsi_soal WHERE soal_id = :soal_id");
                    $opsiStmt->execute(['soal_id' => $soal['soal_id']]);
                    $opsiList = $opsiStmt->fetchAll(PDO::FETCH_COLUMN);
                    $opsiList[] = $soal['jawaban_benar'];
                    $soalList[$key]['opsi'] = array_unique($opsiList);
                }
            }
            return $soalList;
        } catch (PDOException $e) {
            return null;
        }
    }

    public function getUjianIdFromHasilId($hasilId) {
        try {
            $stmt = $this->db->prepare("SELECT ujian_id FROM hasil_ujian WHERE hasil_id = :hasil_id");
            $stmt->execute(['hasil_id' => $hasilId]);
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return null;
        }
    }
}
?>