# api-backend/main.py

import os
import vertexai
from vertexai.generative_models import GenerativeModel, Part
from flask import Flask, request, jsonify
from flask_cors import CORS
import PyPDF2
import docx
import io
import json

# --- Konfigurasi dan Inisialisasi ---
# Ganti dengan Project ID & Lokasi Google Cloud Anda
PROJECT_ID = "ujian-otomatis-api-prod"
LOCATION = "asia-southeast1" # Contoh: us-central1, asia-southeast1

# Pastikan file credentials.json ada di direktori yang sama
os.environ['GOOGLE_APPLICATION_CREDENTIALS'] = 'credentials.json'

# Inisialisasi Vertex AI
vertexai.init(project=PROJECT_ID, location=LOCATION)

app = Flask(__name__)
# Izinkan request dari domain manapun (untuk kemudahan saat development & deploy)
CORS(app)

# Muat model Gemini
model_multimodal = GenerativeModel("gemini-1.5-flash-002")

print("Koneksi ke Vertex AI Gemini 1.5 Flash berhasil.")


# --- Rute API (Sudah Diperbaiki) ---

@app.route('/')
def index():
    """Endpoint dasar untuk mengecek apakah API berjalan."""
    return jsonify({"status": "API Backend AI v3.1 (Fixed) berjalan!"})

@app.route('/1-extract-text', methods=['POST'])
def extract_text_from_file():
    """
    Langkah 1 & 2: Menerima file (gambar/PDF/DOCX), dan langsung mengekstrak teks
    menggunakan Gemini tanpa menyimpan file ke disk.
    """
    if 'file' not in request.files:
        return jsonify({"error": "File tidak ditemukan dalam request"}), 400

    file = request.files['file']
    
    # Baca seluruh konten file ke dalam memori sebagai bytes
    file_bytes = file.read()
    
    try:
        # Siapkan konten untuk dikirim ke model Gemini
        file_part = Part.from_data(data=file_bytes, mime_type=file.mimetype)

        # Prompt untuk Gemini agar fokus pada ekstraksi teks
        prompt = "Anda adalah sistem OCR (Optical Character Recognition) cerdas. Ekstrak semua teks yang terlihat dari file ini seakurat mungkin. Hanya kembalikan teks mentahnya tanpa komentar atau format tambahan."

        # Panggil model Gemini dengan file dan prompt
        response = model_multimodal.generate_content([file_part, prompt])
        extracted_text = response.text

        # Kembalikan hanya teks yang sudah diekstrak
        return jsonify({
            "extracted_text": extracted_text.strip()
        })

    except Exception as e:
        print(f"Error di /1-extract-text: {e}")
        return jsonify({"error": f"Terjadi kesalahan saat memproses file: {str(e)}"}), 500

@app.route('/2-summarize-text', methods=['POST'])
def summarize_text():
    """
    Langkah 3: Menerima teks yang sudah diverifikasi guru dan membuat ringkasan.
    """
    data = request.get_json()
    if not data or 'text' not in data:
        return jsonify({"error": "Request harus berisi 'text' dalam format JSON"}), 400

    verified_text = data['text']
    prompt = f"""
    Anda adalah seorang guru ahli. Buatlah ringkasan yang padat dan informatif (sekitar 100-150 kata) dari materi berikut.
    Fokus pada konsep-konsep kunci yang paling penting untuk dijadikan dasar pembuatan soal ujian.
    
    Materi:
    ---
    {verified_text}
    ---
    Ringkasan:
    """
    try:
        response = model_multimodal.generate_content(prompt)
        return jsonify({"summary": response.text.strip()})
    except Exception as e:
        print(f"Error di /2-summarize-text: {e}")
        return jsonify({"error": str(e)}), 500

@app.route('/3-generate-questions', methods=['POST'])
def generate_questions():
    """
    Langkah 4 (Otomatis): Membuat soal Pilihan Ganda atau Esai dari teks.
    """
    data = request.get_json()
    if not data or 'context' not in data or 'question_type' not in data:
        return jsonify({"error": "Request harus berisi 'context' dan 'question_type'"}), 400

    context = data['context']
    q_type = data['question_type']
    num_questions = min(int(data.get('num_questions', 3)), 5)

    prompt = ""
    try:
        if q_type == 'pilihan_ganda':
            prompt = f"""
            Anda adalah ahli pembuat soal. Berdasarkan materi berikut, buatlah {num_questions} soal Pilihan Ganda.
            Materi: "{context}"
            Kembalikan jawaban HANYA dalam format JSON, berupa sebuah list. 
            Setiap item dalam list adalah sebuah objek dengan key: "question" (string), "answer" (string jawaban yang benar), dan "distractors" (list berisi 3 string jawaban pengecoh).
            Contoh: [{{"question": "Ibu kota Indonesia adalah...", "answer": "Jakarta", "distractors": ["Bandung", "Surabaya", "Medan"]}}]
            """
        else: # Esai
            prompt = f"""
            Anda adalah ahli pembuat soal. Berdasarkan materi berikut, buatlah {num_questions} soal Esai.
            Materi: "{context}"
            Kembalikan jawaban HANYA dalam format JSON, berupa sebuah list.
            Setiap item dalam list adalah sebuah objek dengan key: "question" (string pertanyaan esai) dan "answer" (string kunci jawaban singkat untuk guru).
            Contoh: [{{"question": "Jelaskan proses fotosintesis.", "answer": "Proses mengubah cahaya matahari, air, dan karbon dioksida menjadi glukosa (energi) dan oksigen."}}]
            """
        
        response = model_multimodal.generate_content(prompt)
        # Membersihkan output JSON dari model untuk memastikan formatnya benar
        cleaned_response = response.text.strip().replace('```json', '').replace('```', '')
        questions = json.loads(cleaned_response)
        
        return jsonify({"questions": questions})
        
    except Exception as e:
        print(f"Error di /3-generate-questions: {e}")
        # Jika terjadi error, coba tampilkan response mentah dari AI untuk debugging
        if 'response' in locals():
            print(f"Response mentah dari model: {response.text}")
        return jsonify({"error": f"Gagal mem-parsing output dari AI: {str(e)}"}), 500

# Menjalankan Server Flask
if __name__ == '__main__':
    # Port diambil dari variabel lingkungan 'PORT', default ke 8080 jika tidak ada
    app.run(debug=True, host='0.0.0.0', port=int(os.environ.get('PORT', 8080)))